% "Survey density forecast comparison in small samples" 
% by Laura Coroneo, Fabrizio Iacone and Fabio Profumo, 
% International Journal of Forecasting, 2024

% Monte Carlo
% This code replicates Tables 1 and 2, the empirical size of the EPA and FE tests.

clc; clear;

TT = [40 80];               % sample sizes
M = 10000;                  % number of simulations
tsize = 0.05;               % test size
Q = [0 2 4 6];              % dependence
aa = [1/3 1/2 1];           % wce bands exponent (T^aa)
bb = [1/4 1/3 1/2 2/3];     % wpe bands exponent (T^bb)

%% initialise
DM_WCE = NaN(length(Q), M, length(aa), length(TT),2);
reject_fix_b= NaN(length(Q), M, length(aa), length(TT),2);
reject_wce_N = NaN(length(Q), M, length(aa), length(TT),2);

DM_WPE = NaN(length(Q), M, length(bb), length(TT),2);
reject_fix_m = NaN(length(Q), M, length(bb), length(TT),2);
reject_wpe_N = NaN(length(Q), M, length(bb), length(TT),2);

for dd = [1 2] % 1 for EPA test; 2 for FE test

    for  ts= 1:length(TT)
        clear A B a b e1 e2 d ww

        T = TT(ts);

        % WCE bands
        wce_bands = floor(T.^aa);

        % WPE bands
        wpe_bands = floor(T.^bb);

        for qq = 1:length(Q)

            q = Q(qq);

            rng(0,'twister');

            if dd==1
                alpha = 1;
                beta(qq) = alpha;
            elseif dd==2
                alpha = 3/8;
                if q==0
                    beta(qq) = 8/3*alpha;
                else
                    beta(qq) = 8*alpha*((4+3*q)/(12*(q+1)));
                end
            end

            for m = 1 : M

                %% simulate
                
                a = unifrnd(0, alpha/(q+1), [T+q 1]);
                b = unifrnd(0, beta(qq)/(q+1), [T+q 1]);

                for t = 1+q:T+q

                    A(t) = sum(a(t-q:t));
                    B(t) = sum(b(t-q:t));

                    if dd==1
                        e1 = [A(t);-A(t);0];
                        e2 = [0;-B(t);B(t)];
                        d(t,:) =  e1'*e1-e2'*e2;
                    elseif dd==2
                        d(t,:) =  2*A(t)^2-A(t)*B(t);
                    end

                end

                d = d(q+1:end);

                %% test

                for w = 1:length(wce_bands)
                    ww = wce_bands(w);
                    [DM_WCE(qq,m,w,ts,dd), reject_fix_b(qq,m,w,ts,dd), reject_wce_N(qq,m,w,ts,dd)] = dm(d, ww);
                end

                for w =  1:length(wpe_bands)
                    ww = wpe_bands(w);
                    [~,~,~, DM_WPE(qq,m,w,ts,dd), reject_fix_m(qq,m,w,ts,dd), reject_wpe_N(qq,m,w,ts,dd)] = dm(d, w, ww);
                end
            end
        end
    end

%% Tables

format bank
for ts= 1:length(TT)
    printmat_LC(squeeze(sum(reject_wce_N(:,:,:,ts,dd),2))/M, ...
        ['Table ', num2str(dd),'A: WCE, T=', num2str(TT(ts))], num2str(Q), ...
        'WCET3 WCET2 WCET');
    printmat_LC(squeeze(sum(reject_wpe_N(:,:,:,ts,dd),2))/M, ['Table ',num2str(dd),'A: WPE, T=', num2str(TT(ts))], num2str(Q), 'WPET4 WPET3 WPET2 WPET23');
end

for ts= 1:length(TT)
    printmat_LC(squeeze(sum(reject_fix_b(:,:,:,ts,dd),2))/M, ['Table ',num2str(dd),'B: WCE, T=', num2str(TT(ts))], num2str(Q), 'WCET3 WCET2 WCET');
    printmat_LC(squeeze(sum(reject_fix_m(:,:,:,ts,dd),2))/M, ['Table ',num2str(dd),'B: WPE, T=', num2str(TT(ts))], num2str(Q), 'WPET4 WPET3 WPET2 WPET23');
end
end
